//
//  TWAPICollectionViewController.m
//  Xarth
//
//  Created by Auston Stewart on 10/21/13.
//  Copyright (c) 2013 Justin.tv, Inc. All rights reserved.
//

#import "TWAPICollectionViewController.h"
#import "TWHUDView.h"
#import "TWNoContentOverlayView.h"

@interface TWAPICollectionViewController () <UICollectionViewDelegate, UICollectionViewDataSource, UICollectionViewDelegateFlowLayout>
{
	NSUInteger _offset;
	NSUInteger _lastRequestLimit;
	
	NSTimeInterval _lastRefreshInterval;
	UIRefreshControl *_refreshControl;
	UICollectionViewLayout *_collectionViewLayout;
	
	TWNoContentOverlayView *_noContentOverlayView;
}
@end

@implementation TWAPICollectionViewController

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (void)commonInitialization
{
	// Initialize variables
	_offset = 0;
	_contentSection = 0;
	_limit = 16;
	_items = [[NSMutableArray alloc] init];
	_isAtEndOfList = NO;
	_isRequestingAdditionalItems = NO;
	_isRefreshingItems = NO;
	_infiniteScrollingEnabled = YES;
	_lastRefreshInterval = 0.;
	_includeUnplayableContent = NO;
	_displaysNoContentItem = NO;
	_displaysMoreContentItem = NO;
	_isDisplayingNoContentItem = NO;
	_isDisplayingMoreContentItem = NO;
	
	_operationQueue = [[TWOperationQueue alloc] init];
	_operationQueue.maxConcurrentOperationCount = 4;
}

- (id)initWithCollectionViewLayout:(UICollectionViewLayout *)layout
{
	if ((self = [super init])) {
		
		_collectionViewLayout = layout;
		[self commonInitialization];
	}
	
	return self;
}

- (id)initWithTableViewStyle:(TWAPITableViewStyle)style
{
	if ((self = [super init])) {
		
		_tableViewStyle = style;
		[self commonInitialization];
	}
	
	return self;
}

- (void)setTableViewStyle:(TWAPITableViewStyle)tableViewStyle
{
	if (_tableViewStyle != tableViewStyle) {
		
		_tableViewStyle = tableViewStyle;
		[self.tableView reloadData];
	}
}

- (void)requestItemsFromOffset:(NSUInteger)offset limit:(NSUInteger)limit
{
	// Implement in subclasses
}

- (void)requestAdditionalItems
{
	if (!_isRequestingAdditionalItems && !_isRefreshingItems && !_isAtEndOfList) {
		
		_isRequestingAdditionalItems = YES;
		_lastRequestLimit = _limit;
		[self requestItemsFromOffset:_offset limit:_limit];
	}
}

- (void)refreshItems
{
	if (!_isRequestingAdditionalItems && !_isRefreshingItems) {
		
		_lastRequestLimit = _items.count > 0 ? _offset : _limit;
		_offset = 0;
		_isRefreshingItems = YES;
		[self requestItemsFromOffset:_offset limit:_lastRequestLimit];
	}
}

- (void)refreshIfNecessary
{
	if (!_items.count || (_lastRefreshInterval > 0. && ([NSDate timeIntervalSinceReferenceDate] - _lastRefreshInterval) > 60.)) {
		
		[self refreshItems];
	}
}

- (void) viewWillAppear:(BOOL) animated {
	[super viewWillAppear:animated];
	
	[self refreshIfNecessary];
}

- (void)applicationDidBecomeActive:(NSNotification *)notification
{
	[self refreshIfNecessary];
}

- (void)loadView
{
	UIView *rootView = nil;
	
	if (_collectionViewLayout) {
		
		_collectionView = [[UICollectionView alloc] initWithFrame:[UIScreen mainScreen].bounds collectionViewLayout:_collectionViewLayout];
		_collectionView.delegate = self;
		_collectionView.dataSource = self;
		rootView = _collectionView;
	}
	else {
		
		_tableView = [[UITableView alloc] initWithFrame:[UIScreen mainScreen].bounds style:UITableViewStylePlain];
		_tableView.delegate = self;
		_tableView.dataSource = self;
		rootView = _tableView;
	}
	
	self.view = rootView;
}

- (void)viewDidLoad
{
    [super viewDidLoad];
	
	// Remove the 'Back' text from the back button
	self.navigationItem.backBarButtonItem = [[UIBarButtonItem alloc] initWithTitle:@"" style:UIBarButtonItemStylePlain target:nil action:nil];
	
	// Do any additional setup after loading the view.
	self.view.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
	self.view.backgroundColor = [UIColor whiteColor];
	
	// Instantiate refresh control
	_refreshControl = [[UIRefreshControl alloc] init];
	_refreshControl.tintColor = [UIColor lightGrayColor];
	[_refreshControl addTarget:self action:@selector(refreshControlAction:) forControlEvents:UIControlEventValueChanged];
	
	if (_collectionView) {
		
		[_collectionView addSubview:_refreshControl];
		_collectionView.alwaysBounceVertical = YES;
	}
	else if (_tableView) {
		
		[_tableView addSubview:_refreshControl];
	}
	
	// Register for notifications
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationDidBecomeActive:) name:UIApplicationDidBecomeActiveNotification object:nil];
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

#pragma mark Collection View Delegate Methods

- (void)collectionView:(UICollectionView *)collectionView didSelectItemAtIndexPath:(NSIndexPath *)indexPath
{
	// Implement in subclasses
}

#pragma mark Collection View Data Source

- (NSInteger)numberOfSectionsInCollectionView:(UICollectionView *)collectionView
{
	return 1;
}

- (NSInteger)collectionView:(UICollectionView *)collectionView numberOfItemsInSection:(NSInteger)section
{
	if (section == _contentSection)
		return _numberOfItemsInContentSection;
	
	return 0;
}

- (UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath
{
	// Implement in subclasses
	return nil;
}

- (UICollectionReusableView *)collectionView:(UICollectionView *)collectionView viewForSupplementaryElementOfKind:(NSString *)kind atIndexPath:(NSIndexPath *)indexPath
{
	// FIXME: Dequeue the bottom loading spinner for infinite-scrolling lists
	
	UICollectionReusableView *view = [collectionView dequeueReusableSupplementaryViewOfKind:kind withReuseIdentifier:@"TWInfiniteScrollSpinner" forIndexPath:indexPath];
	
	return view;
}

#pragma mark UITableViewDatasource Methods

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
	return 1;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
	if (section == _contentSection)
		return _numberOfItemsInContentSection;
	
	return 0;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
	// Implement in subclasses
	return nil;
}

#pragma mark API Request Handling

- (BOOL)shouldProcessResponseWithTags:(NSDictionary *)requestTags
{
	return YES;
}

- (void) requestDidFinish:(NSNotification *) notification {
	
	// NSLog(@"In requestDidFinish");
	if (![self shouldProcessResponseWithTags:notification.userInfo[TWAPIRequestRequestTagsKey]])
		return;
	
	BOOL wasRefreshRequest = _isRefreshingItems;
	BOOL wasDisplayingNoContentItem = _isDisplayingNoContentItem;
	BOOL wasDisplayingMoreContentItem = _isDisplayingMoreContentItem;
    NSInteger section = _contentSection;
	
	[_refreshControl endRefreshing];
	
	NSMutableArray *newList = notification.object;
	if (![newList isKindOfClass:[NSArray class]]) {
		_isRefreshingItems = _isRequestingAdditionalItems = NO;
		return;
	}
	
	if (_comparator)
		[newList sortUsingComparator:_comparator];
	
	_offset += _lastRequestLimit;
	_lastRefreshInterval = [NSDate timeIntervalSinceReferenceDate];
	
	NSMutableArray *indexPathsToRemove = [NSMutableArray array];
	NSMutableArray *indexPathsToAdd = [NSMutableArray array];
	NSMutableArray *indexPathsToRefresh = [NSMutableArray array];
	
	// If a No Content item was displayed, remove it
	if (wasDisplayingNoContentItem) [indexPathsToRemove addObject:[NSIndexPath indexPathForItem:0 inSection:section]];
	
	// If a More Content item was displayed, remove it
	if (wasDisplayingMoreContentItem) [indexPathsToRemove addObject:[NSIndexPath indexPathForItem:_items.count inSection:section]];
	
    if (wasRefreshRequest) {
        
        // If this was a refresh request we must remove all items that do not appear in the refreshed list
        if (newList.count > _items.count) {
            
            // More than before, add additional indices, refresh earlier ones
            [indexPathsToAdd addObjectsFromArray:[NSIndexPath indexPathsFromIndex:_items.count forCount:(newList.count - _items.count) inSection:section]];
            [indexPathsToRefresh addObjectsFromArray:[NSIndexPath indexPathsFromIndex:0 forCount:_items.count inSection:section]];
        }
        else if (newList.count < _items.count) {
			
            // Fewer than before, remove additional indices, refresh remaining
            [indexPathsToRemove addObjectsFromArray:[NSIndexPath indexPathsFromIndex:newList.count forCount:(_items.count - newList.count) inSection:section]];
            [indexPathsToRefresh addObjectsFromArray:[NSIndexPath indexPathsFromIndex:0 forCount:newList.count inSection:section]];
        }
        else {
            
            // Same number as before, simply refresh all indices
            [indexPathsToRefresh addObjectsFromArray:[NSIndexPath indexPathsFromIndex:0 forCount:newList.count inSection:section]];
        }
        
        // Remove all objects from _items
        [_items removeAllObjects];
    }
    else {
        
        NSMutableDictionary *existingItems = [NSMutableDictionary dictionaryWithCapacity:(_items ? _items.count : 0)];
        for (NSUInteger i = 0; i < _items.count; i++) {
            NSString *key = [NSString stringWithFormat:@"%d", [_items[i] hash]];
            [existingItems setValue:@(i) forKey:key];
        }
		
        NSMutableIndexSet *indexesToRemoveFromNewList = [NSMutableIndexSet indexSet];
		
        for (NSUInteger i = 0; i < newList.count; i++) {
            NSString *key = [NSString stringWithFormat:@"%d", [newList[i] hash]];
            NSNumber *viewIndex = existingItems[key];
			
            if (!viewIndex) continue;
            NSUInteger index = viewIndex.unsignedIntValue;
			
			// If it exists in the current array, update it in the current list and remove it from the additions
			NSIndexPath *indexPath = [NSIndexPath indexPathForRow:index inSection:section];
			[_items replaceObjectAtIndex:index withObject:newList[i]];
			[indexesToRemoveFromNewList addIndex:i];
			[indexPathsToRefresh addObject:indexPath];
        }
        
        [newList removeObjectsAtIndexes:indexesToRemoveFromNewList];
        [indexPathsToAdd addObjectsFromArray:[NSIndexPath indexPathsFromIndex:_items.count forCount:newList.count inSection:section]];
    }
	
	[_items addObjectsFromArray:newList];
	
	_isAtEndOfList = _items.count < [notification.userInfo[TWAPIRequestSerializedResponseKey][@"_total"] unsignedIntegerValue] ? NO : YES;
	
	// If we show a No Content item, and there are no items, add a row for it
	if (_displaysNoContentItem && _items.count == 0) {
		
		_isDisplayingNoContentItem = YES;
		[indexPathsToAdd addObject:[NSIndexPath indexPathForItem:0 inSection:section]];
	}
	else _isDisplayingNoContentItem = NO;
	
	// If we show a More Content item and we are not at the end of the list, add a row for it
	if (_displaysMoreContentItem && !_isAtEndOfList) {
		
		_isDisplayingMoreContentItem = YES;
		[indexPathsToAdd addObject:[NSIndexPath indexPathForItem:_items.count inSection:section]];
	}
	else _isDisplayingMoreContentItem = NO;
	
	NSInteger numberOfItemsAfterUpdate = _isDisplayingNoContentItem ? 1 : (_isDisplayingMoreContentItem ? _items.count + 1 : _items.count);
	
	// NSLog(@"Will refresh %d index paths",indexPathsToRefresh.count);
	if (_collectionView) {
		
		[_collectionView performBatchUpdates:^{
			
			_numberOfItemsInContentSection = numberOfItemsAfterUpdate;
			[_collectionView deleteItemsAtIndexPaths:indexPathsToRemove];
			[_collectionView insertItemsAtIndexPaths:indexPathsToAdd];
			[_collectionView reloadItemsAtIndexPaths:indexPathsToRefresh];
			
		} completion:^(BOOL completed){
			
			_isRequestingAdditionalItems = NO;
			_isRefreshingItems = NO;
			[self updateViewForContentChange];
		}];
	}
	else if (_tableView) {
		
		_numberOfItemsInContentSection = numberOfItemsAfterUpdate;
		[_tableView beginUpdates];
		[_tableView deleteRowsAtIndexPaths:indexPathsToRemove withRowAnimation:UITableViewRowAnimationMiddle];
		[_tableView insertRowsAtIndexPaths:indexPathsToAdd withRowAnimation:UITableViewRowAnimationMiddle];
		[_tableView reloadRowsAtIndexPaths:indexPathsToRefresh withRowAnimation:UITableViewRowAnimationNone];
		[_tableView endUpdates];
		
		_isRequestingAdditionalItems = NO;
		_isRefreshingItems = NO;
		[self updateViewForContentChange];
	}
}

- (void) requestDidFail:(NSNotification *) notification {
	// NSLog(@"In requestDidFail");
	if (![self shouldProcessResponseWithTags:notification.userInfo[TWAPIRequestRequestTagsKey]])
		return;
	
    _isRefreshingItems = _isRequestingAdditionalItems = NO;
	
	// Stop refresh animation
	[_refreshControl endRefreshing];
	
	if ([notification.object code] == kCFURLErrorNotConnectedToInternet) {
		
		[TWHUDView showErrorHUDWithText:TKLocalizedString(@"No Internet", @"No Internet")];
	}
	else if ([notification.object code] / 100 == 5) {
		
		// Notify user of server error
		[TWHUDView showErrorHUDWithText:TKLocalizedString(@"Server Error", @"Server Error message")];
	}
}

#pragma mark UIScrollViewDelegate methods

- (void) requestMoreItemsOnScrollIfNecessary {
	
	NSArray *sortedIndexPaths = nil;
	
	if (!self.infiniteScrollingEnabled || _isAtEndOfList)
		return;
	
	// Would you believe the index paths aren't sorted?
	if (_collectionView) {
		
		sortedIndexPaths = [[_collectionView indexPathsForVisibleItems] sortedArrayUsingComparator:^(id one, id two) {
			NSIndexPath *pathOne = (NSIndexPath *)one;
			NSIndexPath *pathTwo = (NSIndexPath *)two;
			
			if (pathOne.section < pathTwo.section)
				return NSOrderedAscending;
			if (pathOne.section > pathTwo.section)
				return NSOrderedDescending;
			else {
				
				if (pathOne.row < pathTwo.row)
					return NSOrderedAscending;
				if (pathOne.row > pathTwo.row)
					return NSOrderedDescending;
				return NSOrderedSame;
			}
		}];
	}
	else sortedIndexPaths = [_tableView indexPathsForVisibleRows];
	
	if (([[sortedIndexPaths lastObject] section] == _contentSection) && [[sortedIndexPaths lastObject] row] > (NSInteger)_items.count - 4) {
		
		// If we've scrolled to the bottom, request more items
		[self requestAdditionalItems];
	}
}

- (void) scrollViewDidEndDragging:(UIScrollView *) scrollView willDecelerate:(BOOL) decelerate {
	
    if (!decelerate) {
		
        [self requestMoreItemsOnScrollIfNecessary];
	}
}

- (void) scrollViewDidEndDecelerating:(UIScrollView *) scrollView {
	
	[self requestMoreItemsOnScrollIfNecessary];
}

- (void)refreshControlAction:(id)sender
{
	[self refreshItems];
}

- (void)updateViewForContentChange
{
	// Implement in subclasses
}

#pragma mark No Content Overlay View

- (TWNoContentOverlayView *)noContentOverlayView
{
	if (!_noContentOverlayView) {
		
		_noContentOverlayView = [[TWNoContentOverlayView alloc] initWithFrame:CGRectMake(0.f, 0.f, self.view.frame.size.width, self.view.frame.size.height)];
		[self.view addSubview:_noContentOverlayView];
	}
	
	return _noContentOverlayView;
}

@end
